#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <sys/eventfd.h>

#define THREAD_SIGNAL_NEW_JOB 		0x01
#define THREAD_SHUTDOWN_REQUEST 	0x02

#include "control/worker_thread_set.h"

#include "util/logger.h"

typedef enum 
{
	THREAD_STATE_FREE=0,
	THREAD_STATE_RUNNING,
	THREAD_STATE_DONE
} worker_thread_state_t;

typedef struct worker_thread_t
{
	error_code_t result;
	kmodule_t *module;
	worker_thread_state_t state;
	kmodule_loaded_callback_t callback_func;
	int thread_poll_fd;
	pthread_t pthread_id;
} worker_thread_t;

typedef struct worker_thread_set_t worker_thread_set_t;


typedef struct worker_thread_set_t
{  
	unsigned int active_worker_threads_cnt;

	unsigned int max_worker_threads_cnt;

	worker_thread_t *worker_threads;

	int event_poll_fd;
} worker_thread_set_t;

static worker_thread_set_t wts;

static void worker_thread_set_signal_finished(worker_thread_t *thread);

static worker_thread_t *worker_thread_set_find_free_thread(void);

static error_code_t worker_thread_set_finalize_job(worker_thread_t *thread);

static error_code_t worker_thread_set_wakeup_thread(worker_thread_t *thread);

static void* worker_thread_set_mainloop(void *params);

static void worker_thread_set_stop_threads(void);

//----------------------------------------------- worker thread api members ----------------------------------------------
error_code_t worker_thread_set_init(unsigned int max_worker_threads_cnt)
{
	error_code_t result=RESULT_OK;
	unsigned int i;

	wts.max_worker_threads_cnt=max_worker_threads_cnt;
	wts.worker_threads=malloc(sizeof(worker_thread_t)*max_worker_threads_cnt);
	if (wts.worker_threads!=NULL)
	{
		//RESULT_OK=0, NULL, THREAD_STATE_FREE=0,NULL
		memset(wts.worker_threads,0,sizeof(worker_thread_t)*max_worker_threads_cnt);

		for (i=0;i<max_worker_threads_cnt;i++)
			// -1 means that the workerthreads is not yet running, we are lazy starting the main loop
			wts.worker_threads[i].thread_poll_fd=-1;

		wts.active_worker_threads_cnt=0;
		wts.max_worker_threads_cnt=max_worker_threads_cnt;
	}
	else
		result=RESULT_NORESOURCES;

	if (result==RESULT_OK)
	{
		//start with event flag reset, WORK_BLOCKING, NOT SEMAPHOREMODE, NO CLOEXEC
		wts.event_poll_fd=eventfd(0,0);
		if (wts.event_poll_fd==-1)
			result=RESULT_NORESOURCES;
	}

	return result;
}

void worker_thread_set_deinit(void)
{
	if (wts.worker_threads!=NULL)
	{
		worker_thread_set_stop_threads();
		free(wts.worker_threads);
	}

	if (wts.event_poll_fd!=-1)
		close(wts.event_poll_fd);
}

bool worker_thread_set_can_schedule_something(void)
{
	return wts.active_worker_threads_cnt<wts.max_worker_threads_cnt;
}

int worker_thread_set_get_pollfd(void)
{
	return wts.event_poll_fd;
}

error_code_t worker_thread_set_on_event(void)
{
	unsigned int i;
	int result;
	error_code_t job_result=RESULT_OK;

	//reset eventfd of the module loader main loop
	uint64_t event_cntr;
	result=read(wts.event_poll_fd,&event_cntr, sizeof(uint64_t));

	//ignoring it when we have not been clearly from the threads
	if (result!=(int)sizeof(uint64_t))
		return RESULT_OK;

	for (i=0;i<wts.max_worker_threads_cnt && job_result==RESULT_OK;i++)
	{
		worker_thread_t *thread=&wts.worker_threads[i];
		if (thread->state == THREAD_STATE_DONE)
			job_result=worker_thread_set_finalize_job(thread);
	}

	return job_result;
}

error_code_t worker_thread_set_start_loading_module(kmodule_t *kmodule, kmodule_loaded_callback_t callback_func)
{
	worker_thread_t *thread;

	logger_log_debug("WORKER_THREAD_SET -> Going to load module %s now.",kmodule_get_name(kmodule));
	thread = worker_thread_set_find_free_thread();
	if (thread==NULL)
		return RESULT_INVALID;

	thread->module=kmodule;
	thread->callback_func=callback_func;
	thread->state=THREAD_STATE_RUNNING;

	wts.active_worker_threads_cnt++;

	logger_log_debug("WORKER_THREAD_SET -> Found a thread for the module and waking it up.");

	return worker_thread_set_wakeup_thread(thread);
}
//------------------------------------------------------------------------------------------------------------------------

//----------------------------------------------- worker thread set private members ------------------------------------------
static void worker_thread_set_signal_finished(worker_thread_t *thread)
{
	uint64_t event_incr=1;
	thread->state=THREAD_STATE_DONE;
	if (write(wts.event_poll_fd,&event_incr,sizeof(uint64_t)))
	{
	}
}

static worker_thread_t *worker_thread_set_find_free_thread(void)
{
	unsigned int i;
	worker_thread_t *thread=NULL;

	for (i=0;i<wts.max_worker_threads_cnt;i++)
	{
		if (wts.worker_threads[i].state==THREAD_STATE_FREE)
		{
			thread=&wts.worker_threads[i];
			break;
		}
	}

	return thread;
}

static error_code_t worker_thread_set_wakeup_thread(worker_thread_t *thread)
{
	uint64_t event_cntr=THREAD_SIGNAL_NEW_JOB;

	//if thread is not yet running, kick it off (lazy starting)
	if (thread->thread_poll_fd==-1)
	{
		logger_log_debug("WORKER_THREAD_SET -> Mainloop of thread has not been started before. Doing it now ...");
		//start with event flag reset, WORK_BLOCKING, NOT SEMAPHOREMODE, NO CLOEXEC
		thread->thread_poll_fd=eventfd(0,0);
		if (thread->thread_poll_fd==-1)
			return RESULT_NORESOURCES;

		if (pthread_create(&thread->pthread_id,NULL,worker_thread_set_mainloop, thread)!=0)
		{
			close(thread->thread_poll_fd);
			thread->thread_poll_fd=-1;
			return RESULT_NORESOURCES;
		}
		logger_log_debug("WORKER_THREAD_SET -> Thread is now running.");
	}

	if (write(thread->thread_poll_fd,&event_cntr,sizeof(uint64_t))!=(int)sizeof(uint64_t))
		//don't need to clean up, exiting anyway
		return RESULT_NORESOURCES;

	logger_log_debug("WORKER_THREAD_SET -> Sent the signal to the thread that a new job is present now.");

	return RESULT_OK;
}

static void* worker_thread_set_mainloop(void *params)
{
	bool loop_active=true;
	worker_thread_t *thread;

	thread=(worker_thread_t *)params;

	// need to setup the pollfd before. If not done, someone used us wrongly
	if (thread->thread_poll_fd==-1)
		return NULL;

	while(loop_active)
	{
		int result;
		uint64_t event_cntr;

		result=read(thread->thread_poll_fd,&event_cntr, sizeof(uint64_t));
		if (result==(int)sizeof(uint64_t) && event_cntr!=0)
		{
			//shutdown request
			if (event_cntr==THREAD_SHUTDOWN_REQUEST)
				loop_active=false;

			//new module scheduled, state of thread must be set to state running
			//before the thread is actually going to work
			if (event_cntr==THREAD_SIGNAL_NEW_JOB && thread->state==THREAD_STATE_RUNNING)
			{
				thread->result=kmodule_load(thread->module);
				thread->state=THREAD_STATE_DONE;
				worker_thread_set_signal_finished(thread);
			}
		}
	}

	close(thread->thread_poll_fd);
	thread->thread_poll_fd=-1;

	return NULL;
}

static error_code_t worker_thread_set_finalize_job(worker_thread_t *thread)
{
	//signal the fsm that the job with the module is sone
	thread->callback_func(thread->module,thread->result);

	//empty the worker thread
	thread->callback_func=NULL;
	thread->module=NULL;
	thread->state=THREAD_STATE_FREE;

	wts.active_worker_threads_cnt--;
	return thread->result;
}

static void worker_thread_set_stop_threads(void)
{
	unsigned int i;

	//sending shutdown requests to all threads
	for (i=0;i<wts.max_worker_threads_cnt;i++)
	{
		if (wts.worker_threads[i].thread_poll_fd!=-1)
		{
			uint64_t request=THREAD_SHUTDOWN_REQUEST;
			//thread is still running, sending a stop request
			logger_log_debug("Sending a stop request to a worker thread.");
			if (write(wts.worker_threads[i].thread_poll_fd,&request,sizeof(uint64_t)))
			{
			}
		}
	}

	logger_log_debug("Waiting for threads to exit.");

	//waiting for them to exit
	for (i=0;i<wts.max_worker_threads_cnt;i++)
	{
		if (wts.worker_threads[i].thread_poll_fd!=-1)
			pthread_join(wts.worker_threads[i].pthread_id,NULL);
	}
}
//------------------------------------------------------------------------------------------------------------------------
